package com.genesyslab.mobile.android.sample;

import java.io.File;
import java.util.ArrayList;
import java.util.List;

import org.apache.http.NameValuePair;
import org.apache.http.message.BasicNameValuePair;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.slf4j.LoggerFactory;

import android.app.AlertDialog;
import android.content.DialogInterface;
import android.content.Intent;
import android.content.SharedPreferences;
import android.content.pm.PackageManager;
import android.net.Uri;
import android.os.Bundle;
import android.preference.PreferenceManager;
import android.util.Log;
import android.widget.Toast;
import ch.qos.logback.classic.LoggerContext;
import ch.qos.logback.classic.android.BasicLogcatConfigurator;
import ch.qos.logback.classic.encoder.PatternLayoutEncoder;
import ch.qos.logback.classic.spi.ILoggingEvent;
import ch.qos.logback.core.FileAppender;


public class GenesysSampleActivity extends AbstractGenesysActivity {
	private SharedPreferences sharedPrefences;
	
	@Override
	protected void onCreate(Bundle savedInstanceState) {
		super.onCreate(savedInstanceState);
		sharedPrefences = PreferenceManager.getDefaultSharedPreferences(this);
		getFragmentManager().beginTransaction()
                .replace(android.R.id.content, new GenesysSampleFragment())
                .commit();
		setupLogging();
	}
	
	@Override
	protected void handleIntent(Intent intent) {
		boolean isStartSession = Globals.ACTION_GENESYS_START_SESSION.equals(intent.getAction());
		if (isStartSession) {
			String serviceNamePreference = intent.getDataString();
			String serviceName = sharedPrefences.getString(serviceNamePreference, null);
			boolean registerCloudMessaging = sharedPrefences.getBoolean("push_notifications_enabled", false);
			
			List<NameValuePair> params = new ArrayList<NameValuePair>();
			boolean requestChat = sharedPrefences.getBoolean("request_chat_media_type", false);
			if (requestChat)
				params.add(new BasicNameValuePair("_media_type", "chat"));
			
			genesysService.startSession(serviceName, params, registerCloudMessaging);
		}
		
		boolean isErrorMessage = Globals.ACTION_GENESYS_ERROR_MESSAGE.equals(intent.getAction());
		if (isErrorMessage) {
			String message = intent.getStringExtra(Globals.EXTRA_MESSAGE);
			Toast.makeText(this, message, Toast.LENGTH_LONG).show();
		}
		
		boolean isResponse = Globals.ACTION_GENESYS_RESPONSE.equals(intent.getAction());
		boolean isCloudMessage = Globals.ACTION_GENESYS_CLOUD_MESSAGE.equals(intent.getAction());		
		if (isResponse || isCloudMessage) {
			String message = intent.getExtras().getString(Globals.EXTRA_MESSAGE);
			try {
				JSONObject messageJson = new JSONObject(message);
				
				if (isResponse)
					interpretResponse(messageJson);
				else
					interpretCloudMessage(messageJson);
			} 
			catch (Exception e) {
				Log.e(Globals.GENESYS_LOG_TAG, "Wrong response", e);
				String toastText = "Unexpected response from the Genesys server, see logs";
				Toast.makeText(GenesysSampleActivity.this, toastText, Toast.LENGTH_LONG).show();
			}
		}
	}

	private void makeCall(Uri telUri) {
		boolean canCall = checkCallingOrSelfPermission("android.permission.CALL_PHONE") == PackageManager.PERMISSION_GRANTED;
		String action = canCall ? Intent.ACTION_CALL : Intent.ACTION_DIAL;
		startActivity(new Intent(action, telUri));
	}
	
	private void interpretResponse(JSONObject response) throws JSONException {
		String action = response.optString("_action", null);

		if ("DialNumber".equals(action)) {
			String telUri = response.getString("_tel_url");
			makeCall(Uri.parse(telUri));
		}
		else if ("ConfirmationDialog".equals(action)) {
			String text = response.getString("_text");
			Toast.makeText(this, text, Toast.LENGTH_LONG).show();
		}
		else if ("DisplayMenu".equals(action)) {
			String label = response.getString("_label");
			JSONArray content = response.getJSONArray("_content");
			JSONObject group = content.getJSONObject(0);
			String groupName = group.getString("_group_name");
			final JSONArray groupContent = group.getJSONArray("_group_content");
			String[] menuItems = new String[groupContent.length()];
			for (int i = 0; i < groupContent.length(); i++) {
				menuItems[i] = groupContent.getJSONObject(i).getString("_label");
			}
			
		    AlertDialog.Builder builder = new AlertDialog.Builder(this);
		    builder.setTitle(label + "\n" + groupName)
	           .setItems(menuItems, new DialogInterface.OnClickListener() {
	               public void onClick(DialogInterface dialog, int which) {
						try {
							String url = groupContent.getJSONObject(which).getString("_user_action_url");
							genesysService.continueDialog(url);
						} catch (JSONException e) {
							throw new RuntimeException(e);
						}
	               }
	           });
		    builder.create().show();
		}
		else if ("StartChat".equals(action)) {
			Intent intent = new Intent(this, GenesysChatActivity.class);
			intent.setAction(Globals.ACTION_GENESYS_START_CHAT);
			intent.putExtra(Globals.EXTRA_CHAT_URL, response.getString("_start_chat_url"));
			intent.putExtra(Globals.EXTRA_COMET_URL, response.getString("_comet_url"));
			intent.putExtra(Globals.EXTRA_SUBJECT, response.getJSONObject("_chat_parameters").getString("subject"));
			startActivity(intent);
		}
	}
	
	private void interpretCloudMessage(JSONObject message) throws JSONException {
		String id = message.getString("_id");
		String action = message.getString("_action");
		String urlHostPort = sharedPrefences.getString("server_url", null);
		String servicePath = sharedPrefences.getString("url_path", null);
		String url = urlHostPort + servicePath  + id + "/" + action;
		genesysService.continueDialog(url);
	}

	private void setupLogging() {
		BasicLogcatConfigurator.configureDefaultContext();
		
		LoggerContext lc = (LoggerContext)LoggerFactory.getILoggerFactory();		
	    PatternLayoutEncoder encoder = new PatternLayoutEncoder();
	    encoder.setContext(lc);
	    encoder.setPattern("%d{HH:mm:ss.SSS} %-5level%n - %msg%n%n");
	    encoder.start();

	    FileAppender<ILoggingEvent> fileAppender = new FileAppender<ILoggingEvent>();
	    fileAppender.setContext(lc);
		String logFile = getCacheDir().getAbsolutePath() + File.separator + "log";
	    fileAppender.setFile(logFile);
	    fileAppender.setEncoder(encoder);
	    fileAppender.start();
	    
	    ch.qos.logback.classic.Logger root = (ch.qos.logback.classic.Logger) LoggerFactory.getLogger(Globals.GENESYS_LOG_TAG);
	    root.addAppender(fileAppender);
	}

}
